"use client";

import React, { useEffect, useState } from "react";
import { Swiper, SwiperSlide } from 'swiper/react';
import { Autoplay, Pagination } from 'swiper/modules';
import 'swiper/css';
import 'swiper/css/pagination';
import InnerBanner from "../../../../../components/InnerBanner";
import { useCart } from "@/context/CartContext";
import { useRouter } from "next/navigation";
import BuyNowGate from "@/components/BuyNowGate";
import { getToken } from "@/lib/apiEndpoints";

type Section = {
    parent_section_headline: string;
    parent_section_description: string;
    subsections: any[];
};

type Coupon = {
    couponName: string;
    couponCode: string;
    couponText: string;
};

type AssignmentDetail = {
    title: string;
    text: string;
};

export interface ProductData {
    code: string;
    title: string;
    breadcrumbTitle: string;
    maxRetailPrice: number;
    salePrice: number;
    assignment_quantity: number;
    discount: number;
    extraDiscount: number;
    offerEndsOn: string;
    assignmentsValidity: string;
    image: string;
    types: any[];
    medium: any[];
    sessions: any[];
    coupons: Coupon[];
    highlightHeadline: string;
    highlights: string[];
    question_paper_text: string;
    short_description: string;
    downloadInfo: string;
    main_description: string;
    assignmentDetails: AssignmentDetail[];
    questionPaperImage: string;
    coverPageInfo: string;
    relatedProducts?: any[];
}

interface SolvedAssignmentDetailClientProps {
    productData: ProductData;
    firstSection: Section | null;
    secondSection: Section | null;
    thirdSection: Section | null;
    courseCode: string;
    assignmentCode: string;
}

export default function SolvedAssignmentDetailClient({
    productData,
    firstSection,
    secondSection,
    thirdSection,
    courseCode,
    assignmentCode
}: SolvedAssignmentDetailClientProps) {
    console.log("productData:", productData);
    console.log("question_paper_text:", productData?.question_paper_text);
    const router = useRouter();
    const isLoggedIn = !!getToken();

    const { addToCart, setBuyNowItem } = useCart();
    const [selectedType, setSelectedType] = useState("");
    const [selectedMedium, setSelectedMedium] = useState("");
    const [selectedSession, setSelectedSession] = useState("");

    const isDisabled = !selectedType || !selectedMedium || !selectedSession;

    // Countdown Timer State
    const [timeLeft, setTimeLeft] = useState({
        days: 2,
        hours: 12,
        minutes: 7,
        seconds: 27,
    });

    // Load FontAwesome and Bootstrap - (Optional based on layout, but keeping for safety as per original)
    useEffect(() => {
        if (!document.querySelector('link[href*="font-awesome"]')) {
            const faLink = document.createElement('link');
            faLink.rel = 'stylesheet';
            // Assuming this is handled globally in layout usually, but keeping logic
            // User had empty hrefs in original code, so this might be redundant or needed cleanup.
            // I'll skip adding empty hrefs to avoid errors and assume Layout handles it or user adds them back if needed.
        }
    }, []);

    const [selectionError, setSelectionError] = useState<string | null>(null);

    const handleBuyNow = () => {
        setSelectionError(null);

        if (!selectedType) {
            setSelectionError("Please select assignment type.");
            return;
        }

        if (!selectedMedium) {
            setSelectionError("Please select medium.");
            return;
        }

        if (!selectedSession) {
            setSelectionError("Please select session.");
            return;
        }

        const rawString = `${courseCode}|${assignmentCode}|${Date.now()}`;

        setBuyNowItem({
            id: `solved-assignment-${rawString}`,
            name: productData?.title ?? "",
            price: productData?.salePrice ?? 0,
            qty: quantity ?? 1,
            img: productData?.image ?? "",
            courseCode,
            itemCode: assignmentCode,
            subjectIds: null,
            selectedType,
            selectedMedium,
            selectedSession,
            dataSaveId: null,
            pageName: "solved-assignment-page",
        });

        // 👉 Redirect to checkout page
        router.push("/checkout");
    };

    const handleAddToCart = () => {
        // Reset error
        setSelectionError(null);

        if (!selectedType) {
            setSelectionError("Please select assignment type.");
            return;
        }

        if (!selectedMedium) {
            setSelectionError("Please select medium.");
            return;
        }

        if (!selectedSession) {
            setSelectionError("Please select session.");
            return;
        }

        const rawString = `${courseCode}|${assignmentCode}|${Date.now()}`;

        addToCart({
            id: `solved-assignment-${rawString}`,
            name: productData?.title ?? "",
            price: productData?.salePrice ?? 0,
            qty: quantity ?? 1,
            img: productData?.image ?? "",
            courseCode,
            itemCode: assignmentCode,
            subjectIds: null,
            selectedType,
            selectedMedium,
            selectedSession,
            dataSaveId: null,
            pageName: "solved-assignment-page"
        });
    };

    useEffect(() => {
        if (!productData?.offerEndsOn) return;

        const end = new Date(productData.offerEndsOn).getTime();

        const timer = setInterval(() => {
            const diff = end - Date.now();

            if (diff <= 0) {
                clearInterval(timer);
                setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
                return;
            }

            setTimeLeft({
                days: Math.floor(diff / (1000 * 60 * 60 * 24)),
                hours: Math.floor((diff / (1000 * 60 * 60)) % 24),
                minutes: Math.floor((diff / (1000 * 60)) % 60),
                seconds: Math.floor((diff / 1000) % 60),
            });
        }, 1000);

        return () => clearInterval(timer);
    }, [productData?.offerEndsOn]);

    // Quantity State
    const [quantity, setQuantity] = useState(1);
    const maxQuantity = productData?.assignment_quantity ?? 1;

    const increaseQty = () =>
        setQuantity(prev => (prev < maxQuantity ? prev + 1 : maxQuantity));

    const decreaseQty = () =>
        setQuantity(prev => (prev > 1 ? prev - 1 : 1));

    useEffect(() => {
        setQuantity(1);
    }, [productData]);

    return (
        <div>
            {/* Styles */}
            <style>{`
                @keyframes slideInDown {
                    from {
                        transform: translateY(-100%);
                        opacity: 0;
                    }
                    to {
                        transform: translateY(0);
                        opacity: 1;
                    }
                }

                @keyframes fadeInUp {
                    from {
                        transform: translateY(30px);
                        opacity: 0;
                    }
                    to {
                        transform: translateY(0);
                        opacity: 1;
                    }
                }

                @keyframes pulse {
                    0%, 100% {
                        transform: scale(1);
                    }
                    50% {
                        transform: scale(1.05);
                    }
                }

                @keyframes bounce {
                    0%, 100% {
                        transform: translateY(0);
                    }
                    50% {
                        transform: translateY(-10px);
                    }
                }

                @keyframes shimmer {
                    0% {
                        background-position: -1000px 0;
                    }
                    100% {
                        background-position: 1000px 0;
                    }
                }

                .animate-slideInDown {
                    animation: slideInDown 0.6s ease-out;
                }

                .animate-fadeInUp {
                    animation: fadeInUp 0.8s ease-out;
                }

                .animate-pulse {
                    animation: pulse 2s infinite;
                }

                .animate-bounce {
                    animation: bounce 1s infinite;
                }

                .hover-lift {
                    transition: all 0.3s ease;
                }

                .hover-lift:hover {
                    transform: translateY(-10px);
                    box-shadow: 0 20px 40px rgba(47, 128, 255, 0.3) !important;
                }

                .btn-hover-scale {
                    transition: all 0.3s ease;
                }

                .prices-sy {
                    font-size: 24px;
                }
                span.h2.fw-bold.d-flex {
                    font-size: 48px;
                }
                .btn-hover-scale:hover {
                    transform: scale(1.05);
                    box-shadow: 0 8px 20px rgba(47, 128, 255, 0.4);
                }

                .gradient-text {
                    background: linear-gradient(135deg, #2f80ff 0%, #000 100%);
                    -webkit-background-clip: text;
                    -webkit-text-fill-color: transparent;
                    background-clip: text;
                }

                .solved-assignment-page .card {
                    position: sticky;
                    top: 142px;
                }
                .shimmer {
                    background: linear-gradient(90deg, transparent, rgba(47, 128, 255, 0.3), transparent);
                    background-size: 1000px 100%;
                    animation: shimmer 2s infinite;
                }

                .card-hover {
                    transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
                }

                .card-hover:hover {
                    transform: translateY(-15px) scale(1.03);
                    box-shadow: 0 25px 50px rgba(47, 128, 255, 0.4);
                }

                .icon-rotate:hover {
                    animation: rotate 0.6s ease;
                }

                @keyframes rotate {
                    from { transform: rotate(0deg); }
                    to { transform: rotate(360deg); }
                }

                .badge-animate {
                    animation: pulse 1.5s infinite;
                }
                .badge-animate {
                    background: red;
                    color: white;
                    padding: 3px 16px;
                    border-radius: 20px;
                    margin-left: 20px;
                }

                .border-glow {
                    box-shadow: 0 0 20px rgba(47, 128, 255, 0.5);
                    transition: all 0.3s ease;
                }

                .border-glow:hover {
                    box-shadow: 0 0 30px rgba(47, 128, 255, 0.8);
                }

                .whatsapp-float {
                    animation: bounce 2s infinite;
                }

                .whatsapp-float:hover {
                    animation: none;
                    transform: scale(1.1);
                }
                .highlight-item {
                    transition: all 0.3s ease;
                    padding: 8px;
                    border-radius: 8px;
                    width: 50% !important;
                    font-size: 12px;
                }

                .highlight-item:hover {
                    background: rgba(47, 128, 255, 0.1);
                    transform: translateX(10px);
                }

                .table-row-hover {
                    transition: all 0.3s ease;
                }

                .table-row-hover:hover {
                    background: rgba(47, 128, 255, 0.05);
                    transform: scale(1.01);
                }

                .step-number {
                    display: inline-flex;
                    align-items: center;
                    justify-content: center;
                    width: 40px;
                    height: 40px;
                    background: linear-gradient(135deg, #2f80ff, #1a5acc);
                    color: white;
                    border-radius: 50%;
                    font-weight: bold;
                    margin-right: 15px;
                    transition: all 0.3s ease;
                }

                .step-number:hover {
                    transform: rotate(360deg) scale(1.2);
                    box-shadow: 0 5px 15px rgba(47, 128, 255, 0.5);
                }
                .selete-box {
                    border-radius: 10px;
                    padding: 0px 20px;
                    display: flex;
                    align-items: center;
                    width: 300px;
                }
            `}</style>

            <InnerBanner title={productData?.title} breadcrumb={productData?.breadcrumbTitle} />

            {/* Main Content */}
            <div className="solved-assignment-page py-5">
                <div className="container">
                    <div className="row g-4 animate-fadeInUp">
                        {/* Left - Product Image */}
                        <div className="col-lg-5">
                            <div className="card border-0 shadow-lg hover-lift border-glow" style={{ borderRadius: '20px', overflow: 'hidden' }}>
                                <div style={{ position: 'relative' }}>
                                    <img
                                        src={productData?.image}
                                        className="card-img-top"
                                        alt={productData?.code}
                                        style={{ borderRadius: '20px 20px 0 0' }}
                                    />
                                    <div
                                        className="badge-animate"
                                        style={{
                                            position: 'absolute',
                                            top: '20px',
                                            right: '20px',
                                            background: '#dc3545',
                                            color: '#fff',
                                            padding: '10px 20px',
                                            borderRadius: '50px',
                                            fontSize: '16px',
                                            fontWeight: 'bold',
                                            boxShadow: '0 5px 15px rgba(220, 53, 69, 0.5)'
                                        }}
                                    >
                                        {productData?.discount}% OFF
                                    </div>
                                </div>
                            </div>
                        </div>

                        {/* Right - Product Details */}
                        <div className="col-lg-7">
                            <h1 className="h3 fw-bold mb-4" style={{ lineHeight: '1.4' }}>
                                {productData?.title}
                            </h1>

                            {/* Type Selector */}
                            <div className=" selete-box mb-1 d-flex gap-3 border ">
                                <label className="form-label d-flex m-0" style={{ color: '#000' }}>
                                    Type
                                </label>
                                <select
                                    className="form-select border-0"
                                    value={selectedType}
                                    onChange={(e) => setSelectedType(e.target.value)}
                                >
                                    <option value="">-- Select Type --</option>
                                    {productData?.types.map((type: any) => (
                                        <option key={type.enc_id} value={type.enc_id}>{type.type_name}</option>
                                    ))}
                                </select>
                            </div>

                            {/* Medium Selector */}
                            <div className="mb-1 selete-box d-flex gap-3 border ">
                                <label className="form-label d-flex m-0" style={{ color: '#000' }}>
                                    Medium
                                </label>
                                <select
                                    className="form-select border-0"
                                    value={selectedMedium}
                                    onChange={(e) => setSelectedMedium(e.target.value)}
                                >
                                    <option value="">-- Select Medium --</option>
                                    {productData?.medium.map((language: any) => (
                                        <option key={language.enc_id} value={language.enc_id}>{language.language_name}</option>
                                    ))}
                                </select>
                            </div>

                            {/* Session Selector */}
                            <div className="selete-box d-flex gap-3 border">
                                <label className="form-label d-flex m-0" style={{ color: '#000' }}>
                                    Session
                                </label>
                                <select
                                    className="form-select border-0"
                                    value={selectedSession}
                                    onChange={(e) => setSelectedSession(e.target.value)}
                                >
                                    <option value="">-- Select Session --</option>
                                    {productData?.sessions.map((session: any) => (
                                        <option key={session.enc_id} value={session.enc_id}>{session.session_name}</option>
                                    ))}
                                </select>
                            </div>

                            {/* Price with Animation */}
                            <div className=" p-3" >
                                <div className="">
                                    <div className="d-flex align-items-center">
                                        <span className="h2 fw-bold d-flex align-items-center gap-2"><div className="prices-sy">₹ </div>{productData?.salePrice}</span>
                                        <span className="text-decoration-line-through text-muted ms-3 h5">₹{productData?.maxRetailPrice}</span>

                                        <div className="badge-animate">{productData?.discount}%{productData && productData.extraDiscount > 0 ? ' + Extra ' + productData?.extraDiscount + '% OFF' : ''} </div>
                                    </div>
                                    <div className="text-muted small mt-1">
                                        <i className="fas fa-shield-alt" style={{ color: '#2f80ff' }}></i> INCLUSIVE OF ALL TAXES
                                    </div>
                                    <div className="mt-4">
                                        <h6 className="text-danger">
                                            Offer ends in&nbsp;
                                            {String(timeLeft.days).padStart(2, '0')} day(s): {String(timeLeft.hours).padStart(2, '0')} hr(s): {String(timeLeft.minutes).padStart(2, '0')} min(s): {String(timeLeft.seconds).padStart(2, '0')} sec(s)
                                        </h6>
                                    </div>

                                    <div className="container py-4">
                                        {/* Coupon Slider */}
                                        {productData && productData.coupons?.length > 0 && (
                                            <div className="mb-3 bg-light rounded-4 overflow-hidden">
                                                <Swiper
                                                    modules={[Autoplay, Pagination]}
                                                    spaceBetween={10}
                                                    slidesPerView={1}
                                                    autoplay={{ delay: 3000, disableOnInteraction: false }}
                                                    pagination={{ clickable: true }}
                                                    loop={true}
                                                    className="pb-4"
                                                >
                                                    {/* Slides */}
                                                    {productData?.coupons.map((coupon, index) => (
                                                        <SwiperSlide key={index}>
                                                            <div className="p-3">
                                                                <div className="d-flex align-items-center justify-content-between">
                                                                    <div>
                                                                        <span className="badge bg-secondary mb-1">{coupon.couponName}</span>
                                                                        <h6 className="fw-bold mb-1">{coupon.couponCode}</h6>
                                                                        <p className="mb-0 small text-muted">{coupon.couponText}</p>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                        </SwiperSlide>
                                                    ))}
                                                </Swiper>
                                            </div>
                                        )}

                                        {/* Deadline Alert */}
                                        <div className="alert alert-light border mt-3 d-flex gap-3">
                                            <span> 🚨 Alert! New deadline for the assignments: {productData?.assignmentsValidity}.</span>
                                        </div>

                                        {/* Quantity Selector */}
                                        <div className="d-flex align-items-center gap-2 my-4">
                                            <button className="btn btn-outline-secondary" onClick={decreaseQty}>-</button>
                                            <span className="fs-4">{String(quantity).padStart(2, '0')}</span>
                                            <button className="btn btn-outline-secondary" onClick={increaseQty}>+</button>
                                        </div>

                                        {selectionError && (
                                            <div className="alert alert-danger mt-3 py-2">
                                                {selectionError}
                                            </div>
                                        )}

                                        <small
                                            className="text-danger my-1 d-block"
                                            style={{ fontSize: "0.72rem", fontStyle: "italic" }}
                                        >
                                            ** Type, Medium, Session, and Others are mandatory fields.
                                        </small>

                                        {/* Buttons */}
                                        <div className="d-flex gap-3">
                                            <BuyNowGate
                                                isLoggedIn={isLoggedIn}
                                                handleBuyNow={handleBuyNow}
                                                isDisabled={isDisabled}
                                            />

                                            <button
                                                className="btn btn-primary w-50 py-2 fw-bold"
                                                onClick={handleAddToCart}
                                                disabled={isDisabled}
                                            >
                                                Add to Cart
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Why Choose with Animation */}
                            <div className=" p-4" style={{ background: 'linear-gradient(135deg, rgba(47, 128, 255, 0.05), rgba(47, 128, 255, 0.1))', borderRadius: '15px', border: '0px solid #2f80ff' }}>
                                <h5 className="fw-bold mb-3" style={{ color: '#000' }}>
                                    <i className="fas fa-star" style={{ color: '#ffa500' }}></i> {productData?.highlightHeadline}
                                </h5>
                                <div className="d-flex flex-wrap">
                                    {productData?.highlights.map((item, idx) => (
                                        <div key={idx} className="highlight-item ">
                                            <i className="fas fa-check-circle" style={{ color: '#2f80ff', fontSize: '18px' }}></i>
                                            <span className="ms-3">{item}</span>
                                        </div>
                                    ))}
                                </div>
                            </div>

                            {/* Last Date Submission */}
                            <div className="alert mt-3" style={{ background: 'linear-gradient(135deg, #ffffffff, #ffffffff)', border: '2px solid #ffffffff', borderRadius: '15px' }}>
                                <h6 className="fw-bold" style={{ color: '#000' }}>
                                    <i className="fas fa-exclamation-triangle" style={{ color: '#ffa500' }}></i> Last Date of Assignment Submission:
                                </h6>
                                <p className="mb-0 small" dangerouslySetInnerHTML={{ __html: productData?.short_description ?? "" }} />
                            </div>

                            <p className="qus-papper-box shadow border border-black p-4 bg-white text-black" dangerouslySetInnerHTML={{ __html: productData?.question_paper_text ?? "" }} />
                            {/* <div className="qus-papper-box shadow border border-black p-4 bg-white ">
                                <img src={productData?.questionPaperImage} alt={productData?.questionPaperImage} />
                            </div> */}


                        </div>
                    </div>

                    <div className="container pt-5" style={{ background: "#f7f7f7" }}>
                        {/* TITLE */}
                        <h3 className="fw-bold mb-3">
                            <i className="fa-solid fa-book-open text-primary me-2"></i>
                            IGNOU {productData?.code} Solved Assignment {productData?.sessions
                                ? productData.sessions.map((session: any) => session.session_name).join(", ")
                                : ""} Session
                        </h3>

                        {/* MAIN DESCRIPTION */}
                        <div className="p-4 bg-white border rounded shadow-sm mb-4" dangerouslySetInnerHTML={{ __html: productData?.main_description ?? "" }} />

                        {/* DOWNLOAD INFO */}
                        <div className="p-4 bg-white border rounded shadow-sm mb-4" dangerouslySetInnerHTML={{ __html: productData?.downloadInfo ?? "" }} />

                        {/* ASSIGNMENT DETAILS */}
                        <div className="p-4 bg-white border rounded shadow-sm mb-4">
                            <h5 className="fw-bold mb-3 text-dark">
                                {productData?.code} Assignment Details
                            </h5>

                            {productData && productData.assignmentDetails.length > 0 && (
                                <div className="d-flex flex-column gap-2">
                                    {productData?.assignmentDetails.map((item, index) => (
                                        <div
                                            key={index}
                                            className={`d-flex py-2 ${index !== productData?.assignmentDetails.length - 1 ? "border-bottom" : ""
                                                }`}
                                        >
                                            <span className="fw-bold" style={{ width: "150px" }}>
                                                {item.title}:
                                            </span>

                                            <span className="text-secondary">
                                                {item.text}
                                            </span>
                                        </div>
                                    ))}
                                </div>
                            )}

                        </div>

                        {/* COVER PAGE INFO & BUTTONS */}
                        <div className="p-4 bg-white border rounded shadow-sm mb-4 text-center">
                            <span dangerouslySetInnerHTML={{ __html: productData?.coverPageInfo ?? "" }} />

                            <div className="d-grid gap-3 justify-content-center">
                                <a
                                    href="/free-download"
                                    className="btn btn-primary fw-bold py-3 px-4 rounded-pill shadow-sm"
                                    style={{
                                        background: "#4169e1",
                                        borderColor: "#4169e1",
                                        fontSize: "1.1rem",
                                        textDecoration: "none",
                                        color: "#fff",
                                        display: "inline-block",
                                    }}
                                >
                                    Download IGNOU Cover Page Guide & Format Instantly
                                </a>
                            </div>
                        </div>

                        {/* PURCHASE STEPS */}
                        <div className="p-4 bg-white border rounded shadow-sm mb-4">
                            <h3 className="fw-bold mb-4 text-secondary">
                                {firstSection?.parent_section_headline}
                            </h3>

                            {firstSection && firstSection?.subsections.length > 0 && (
                                <div className="d-flex flex-column gap-2 text-secondary">
                                    {firstSection?.subsections.map((step, index) => (
                                        <div key={index} className="d-flex gap-3 align-items-start">
                                            <i className={`fa-solid ${step.sec_icon} text-primary mt-1`} />
                                            <span>{step.title}</span>
                                        </div>
                                    ))}
                                </div>
                            )}
                        </div>

                        {/* TERMS & CONDITIONS */}
                        <div className="p-4 bg-white border rounded shadow-sm mb-4">
                            <h3 className="fw-bold text-secondary mb-2">
                                {secondSection?.parent_section_headline}
                            </h3>
                            {secondSection && (
                                <p className="text-secondary small" dangerouslySetInnerHTML={{ __html: secondSection.parent_section_description }} />
                            )}
                        </div>

                        {/* WHAT'S INCLUDED - NEW ICON LAYOUT */}
                        <div className="p-5 bg-white border rounded shadow-sm mb-5">
                            <h3 className="fw-bold mb-5 text-secondary">
                                {thirdSection?.parent_section_headline}
                            </h3>

                            {thirdSection && thirdSection?.subsections.length > 0 && (
                                <div className="row text-center g-4">
                                    {/* Item 1 */}
                                    {thirdSection?.subsections.map((item, idx) => (
                                        <div key={idx} className="col-md-6 col-lg-3">
                                            <div className="mb-3">
                                                <span className="fa-stack fa-2x text-primary">
                                                    <i className={`fa-solid ${item.sec_icon}`}></i>
                                                </span>
                                            </div>
                                            <h6 className="fw-bold mb-1">{item.title}</h6>
                                            <p className="small text-muted" dangerouslySetInnerHTML={{ __html: item.desc }} />
                                        </div>
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}
